<!--Below are the complete files for the enhanced ODGLOBAL landing site with a **fullscreen parallax slideshow** (Option C). Copy each file into your project folder exactly as named and push to your GitHub repo (`techedemy/odgloballimited-site`). -->

# Project structure

odgloballimited-site/
├── index.html
├── styles.css
├── src/
│   ├── main.js
│   ├── components/
│   │   ├── Header.js
│   │   ├── Hero.js
│   │   ├── Slideshow.js
│   │   ├── Services.js
│   │   ├── About.js
│   │   ├── Projects.js
│   │   ├── Team.js
│   │   └── Contact.js
│   └── data/
│       └── team.js
└── assets/
    ├── logo.png
    ├── slide-1.jpg
    ├── slide-2.jpg
    └── slide-3.jpg
```

---

## 1) `index.html`

```html
<!doctype html>
<html lang="en">
  <head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>ODGLOBAL Limited — Energy. Engineering. Excellence.</title>
    <meta name="description" content="ODGLOBAL Limited — Oil & Gas services, procurement, manpower supply, HSE and engineering solutions." />

    <!-- Tailwind CDN -->
    <script src="https://cdn.tailwindcss.com"></script>

    <!-- FontAwesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

    <!-- React + ReactDOM (UMD) -->
    <script crossorigin src="https://unpkg.com/react@18/umd/react.development.js"></script>
    <script crossorigin src="https://unpkg.com/react-dom@18/umd/react-dom.development.js"></script>

    <!-- Babel (so we can keep separate .js files with JSX, no build tools) -->
    <script src="https://unpkg.com/@babel/standalone/babel.min.js"></script>

    <!-- Motion One (lightweight animation library) -->
    <script src="https://cdn.jsdelivr.net/npm/@motionone/dom/dist/motion.min.js"></script>

    <link rel="stylesheet" href="styles.css" />

  </head>
  <body class="bg-[#071021] text-slate-100 antialiased leading-normal">

    <!-- Root for React app -->
    <div id="root"></div>

    <!-- Load components (JSX) - order matters -->
    <script type="text/babel" src="src/components/Header.js"></script>
    <script type="text/babel" src="src/components/Slideshow.js"></script>
    <script type="text/babel" src="src/components/Hero.js"></script>
    <script type="text/babel" src="src/components/Services.js"></script>
    <script type="text/babel" src="src/components/About.js"></script>
    <script type="text/babel" src="src/components/Projects.js"></script>
    <script type="text/babel" src="src/components/Team.js"></script>
    <script type="text/babel" src="src/components/Contact.js"></script>
    <script type="text/babel" src="src/data/team.js"></script>

    <!-- main entry -->
    <script type="text/babel" src="src/main.js"></script>

  </body>
</html>
```

---

## 2) `styles.css`

```css
/* Basic custom styles and parallax helpers */
:root{
  --accent: #f59e0b; /* amber-500 */
}

/* Ensure hero fills viewport and slideshow layers stack */
.hero-root{ position: relative; height: 100vh; overflow: hidden; }
.slideshow-layer{ position: absolute; inset: 0; background-size: cover; background-position: center; will-change: transform, opacity; }
.slideshow-overlay{ position:absolute; inset:0; background: linear-gradient(180deg, rgba(7,16,33,0.55), rgba(7,16,33,0.6)); }

/* Assistant bubble */
.assistant-bubble{ backdrop-filter: blur(6px); background: rgba(255,255,255,0.03); }

/* Simple responsive tweaks */
@media (max-width:768px){
  .hero-title{ font-size: 1.8rem; }
}
```

---

## 3) `src/components/Slideshow.js`

```jsx
// Slideshow.js - Fullscreen parallax slideshow (auto + manual)
const { useState, useEffect, useRef } = React;

function Slideshow(){
  const slides = [
    'assets/slide-1.jpg',
    'assets/slide-2.jpg',
    'assets/slide-3.jpg'
  ];

  const [index, setIndex] = useState(0);
  const containerRef = useRef();

  useEffect(()=>{
    // autoplay
    const t = setInterval(()=> setIndex(i => (i+1) % slides.length), 6000);
    return ()=> clearInterval(t);
  },[]);

  // parallax on mouse move (subtle)
  useEffect(()=>{
    const el = containerRef.current;
    if(!el) return;
    function onMove(e){
      const rect = el.getBoundingClientRect();
      const x = (e.clientX - rect.left) / rect.width - 0.5;
      const y = (e.clientY - rect.top) / rect.height - 0.5;
      // apply small transform to background layers
      el.querySelectorAll('.slideshow-layer').forEach((layer, i)=>{
        const depth = (i - index) * 6; // vary depth
        const tx = x * depth;
        const ty = y * depth;
        layer.style.transform = `translate3d(${tx}px, ${ty}px, 0)`;
      });
    }
    window.addEventListener('mousemove', onMove);
    return ()=> window.removeEventListener('mousemove', onMove);
  },[index]);

  return (
    <div ref={containerRef} className="hero-root">
      {slides.map((s,i)=> (
        <div
          key={s}
          className="slideshow-layer"
          style={{
            backgroundImage:`url(${s})`,
            opacity: i===index?1:0,
            transition:'opacity 1.2s ease, transform 0.8s ease',
            zIndex: i===index?2:1
          }}
        />
      ))}
      <div className="slideshow-overlay" />

      {/* controls */}
      <div className="absolute bottom-8 left-1/2 transform -translate-x-1/2 flex gap-3 z-30">
        {slides.map((_,i)=> (
          <button key={i} onClick={()=>setIndex(i)} className={`w-3 h-3 rounded-full ${i===index? 'bg-white':'bg-white/40'}`} aria-label={`Go to slide ${i+1}`}></button>
        ))}
      </div>
    </div>
  );
}

// attach to global so main loader can import if needed
window.Slideshow = Slideshow;
```

---

## 4) `src/components/Header.js`

```jsx
function Header(){
  return (
    <header className="fixed w-full z-40 top-0 left-0">
      <div className="max-w-7xl mx-auto px-6 py-4 flex items-center justify-between">
        <div className="flex items-center gap-4">
          <img src="assets/logo.png" alt="OD" className="w-12 h-12 rounded-full object-cover" />
          <div>
            <h1 className="text-lg font-semibold">ODGLOBAL Limited</h1>
            <p className="text-xs text-slate-300">Oil & Gas • Procurement • Engineering • HSE</p>
          </div>
        </div>

        <nav className="hidden md:flex gap-6 items-center text-sm">
          <a href="#about" className="hover:text-amber-400">About</a>
          <a href="#services" className="hover:text-amber-400">Services</a>
          <a href="#projects" className="hover:text-amber-400">Projects</a>
          <a href="#team" className="hover:text-amber-400">Team</a>
          <a href="#contact" className="px-4 py-2 rounded-md bg-amber-500 text-[#071021] font-semibold">Contact</a>
        </nav>

        <div className="md:hidden">
          <button className="px-3 py-2 bg-white/5 rounded">Menu</button>
        </div>
      </div>
    </header>
  );
}

window.Header = Header;
```

---

## 5) `src/components/Hero.js`

```jsx
function Hero(){
  return (
    <section className="relative">
      {/* Slideshow sits full-screen */}
      <Slideshow />

      {/* Hero content placed over slideshow */}
      <div className="absolute inset-0 z-20 flex flex-col items-center justify-center text-center px-6">
        <img src="assets/logo.png" className="w-28 mb-4" alt="ODGLOBAL"/>
        <h2 className="text-4xl md:text-6xl font-extrabold leading-tight hero-title">ODGLOBAL — Your partner for oil & energy solutions</h2>
        <p className="mt-6 text-slate-200 max-w-2xl">We supply engineering equipment, manpower, HSE expertise and end-to-end project support to upstream oil & gas, industrial and infrastructure clients — delivering quality, on time and on budget.</p>
        <div className="mt-8 flex gap-4">
          <a href="#contact" className="px-6 py-3 rounded-md bg-amber-500 text-[#071021] font-semibold shadow">Request a Quote</a>
          <a href="#services" className="px-6 py-3 rounded-md border border-white/10">Our Services</a>
        </div>
      </div>

      {/* small assistant bubble bottom-left */}
      <div className="absolute left-6 bottom-6 z-30 assistant-bubble p-4 rounded-2xl shadow-lg">
        <div className="flex items-start gap-3">
          <div className="w-12 h-12 rounded-full bg-amber-500 flex items-center justify-center font-bold text-[#071021]">OD</div>
          <div>
            <div className="text-sm font-semibold">ODGLOBAL Assistant</div>
            <div className="text-xs text-slate-300">Need project support? Click to start a quick enquiry.</div>
          </div>
        </div>
      </div>
    </section>
  );
}

window.Hero = Hero;
```

---

## 6) `src/components/Services.js`

```jsx
function Services(){
  const items = [
    {title:'Procurement Services', desc:'End-to-end sourcing, MTO review, OEM supply chain & logistics.'},
    {title:'Engineering & Maintenance', desc:'Equipment installation, commissioning, flowlines, valves and corrosion control.'},
    {title:'Manpower Supply', desc:'Qualified personnel: PMs, engineers, HSE, welders, technicians.'}
  ];

  return (
    <section id="services" className="py-16 bg-[#071021]">
      <div className="max-w-7xl mx-auto px-6">
        <h3 className="text-2xl font-semibold mb-6">What we do</h3>
        <div className="grid md:grid-cols-3 gap-6">
          {items.map((it,i)=> (
            <div key={i} className="p-6 rounded-xl bg-white/3 card-hover">
              <h4 className="font-semibold text-lg">{it.title}</h4>
              <p className="mt-3 text-slate-300 text-sm">{it.desc}</p>
            </div>
          ))}
        </div>
      </div>
    </section>
  );
}

window.Services = Services;
```

---

## 7) `src/components/About.js`

```jsx
function About(){
  return (
    <section id="about" className="py-16 bg-gradient-to-b from-transparent to-white/2">
      <div className="max-w-7xl mx-auto px-6 grid md:grid-cols-2 gap-8 items-center">
        <div>
          <h3 className="text-2xl font-semibold">About ODGLOBAL Limited</h3>
          <p className="mt-4 text-slate-300">ODGLOBAL Limited is a Nigerian indigenous company (incorporated Nov 2018) delivering value-added services across Oil & Gas, Procurement, Manpower supply, Engineering tools and Haulage. Our mission is to provide quality services aligned to client specifications while maintaining HSE excellence and operational integrity.</p>

          <ul className="mt-4 list-disc pl-5 text-slate-300 text-sm space-y-2">
            <li>Supplies: Mechanical, piping, electrical, instrumentation, structural steel and safety equipment.</li>
            <li>Strong local content: training, capacity building and technology transfer.</li>
            <li>Management commitment to HSE and consequence management.</li>
          </ul>
        </div>

        <div className="p-6 rounded-xl bg-white/3">
          <h4 className="font-semibold">Mission</h4>
          <p className="text-slate-300 text-sm mt-2">To provide quality engineering equipment, labour supply and construction services that conform with customer requirements.</p>

          <h4 className="font-semibold mt-4">Vision</h4>
          <p className="text-slate-300 text-sm mt-2">To be among the leading Energy and Oil & Gas service providers to international oil companies.</p>

          <h4 className="font-semibold mt-4">Values</h4>
          <div className="mt-2 text-slate-300 text-sm">
            <div><strong>Integrity:</strong> Highest level of business ethics.</div>
            <div className="mt-1"><strong>HSE:</strong> Safety-first operations and training.</div>
            <div className="mt-1"><strong>Teamwork & Excellence:</strong> On time, on budget delivery.</div>
          </div>
        </div>
      </div>
    </section>
  );
}

window.About = About;
```

---

## 8) `src/components/Projects.js`

```jsx
function Projects(){
  return (
    <section id="projects" className="py-16 bg-white/2">
      <div className="max-w-7xl mx-auto px-6">
        <h3 className="text-2xl font-semibold mb-6">Past Projects & Clients</h3>
        <div className="grid md:grid-cols-3 gap-6">
          <div className="p-6 rounded-xl bg-white/3">
            <h4 className="font-semibold">Construction & Supply</h4>
            <p className="text-slate-300 text-sm mt-2">Vosky Pod Venture Limited — building & electrical supply projects (2018–2019)</p>
          </div>
          <div className="p-6 rounded-xl bg-white/3">
            <h4 className="font-semibold">Trusted By</h4>
            <p className="text-slate-300 text-sm mt-2">Shell, TotalE&P, SEPLAT, NLNG, Mobil, NDDC and more.</p>
          </div>
          <div className="p-6 rounded-xl bg-white/3">
            <h4 className="font-semibold">Sewage & Water Treatment</h4>
            <p className="text-slate-300 text-sm mt-2">Containerized plants, RO and membrane technologies for industrial clients.</p>
          </div>
        </div>
      </div>
    </section>
  );
}

window.Projects = Projects;
```

---

## 9) `src/data/team.js`

```jsx
// simple data file (attaches to window)
window.ODTeam = [
  {name:'Umukoro Odiri Prince', role:'President / CEO'},
  {name:'Abiye George', role:'Head, Operations'},
  {name:'Otiotio Aliu', role:'Procurement Specialist'},
  {name:'Ademola Jide', role:'CFO'},
  {name:'Richard Abbey', role:'HSE Officer'},
  {name:'Akinsanya Olanrewaju', role:'Chief Technical Lead'},
  {name:'Denis Ray Ogosi', role:'Regional Manager (Europe)'}
];
```

---

## 10) `src/components/Team.js`

```jsx
function Team(){
  const members = window.ODTeam || [];
  return (
    <section id="team" className="py-16">
      <div className="max-w-7xl mx-auto px-6">
        <h3 className="text-2xl font-semibold mb-6">Management Team</h3>
        <div className="grid md:grid-cols-3 gap-6">
          {members.map((m,i)=> (
            <div key={i} className="p-5 rounded-lg bg-white/3">
              <div className="flex items-center gap-4">
                <div className="w-14 h-14 rounded-full bg-amber-500 flex items-center justify-center font-bold">{m.name.split(' ').map(n=>n[0]).slice(0,2).join('')}</div>
                <div>
                  <div className="font-semibold">{m.name}</div>
                  <div className="text-xs text-slate-300">{m.role}</div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  );
}

window.Team = Team;
```

---

## 11) `src/components/Contact.js`

```jsx
function Contact(){
  return (
    <section id="contact" className="py-16">
      <div className="max-w-7xl mx-auto px-6 grid md:grid-cols-2 gap-8 items-start">
        <div>
          <h3 className="text-2xl font-semibold">Get in touch</h3>
          <p className="mt-4 text-slate-300">For procurement, project enquiries or partnerships — send us a message and our team will respond promptly.</p>

          <div className="mt-6 space-y-3 text-sm text-slate-300">
            <div><strong>Head Office:</strong> Lagos, Nigeria</div>
            <div><strong>Email:</strong> info@odglobal.ng</div>
            <div><strong>Europe Rep:</strong> Cavan, Republic of Ireland</div>
          </div>

        </div>

        <form className="p-6 rounded-xl bg-white/3">
          <label className="block text-xs text-slate-300">Full Name</label>
          <input className="w-full mt-2 p-3 rounded bg-transparent border border-white/10 text-sm" placeholder="Jane Doe" />

          <label className="block text-xs text-slate-300 mt-3">Email</label>
          <input className="w-full mt-2 p-3 rounded bg-transparent border border-white/10 text-sm" placeholder="you@company.com" />

          <label className="block text-xs text-slate-300 mt-3">Message</label>
          <textarea className="w-full mt-2 p-3 rounded bg-transparent border border-white/10 text-sm" rows="4" placeholder="Tell us about your project..."></textarea>

          <div className="mt-4 flex justify-end">
            <button type="button" className="px-4 py-2 rounded bg-amber-500 text-[#071021] font-semibold">Send Message</button>
          </div>
        </form>
      </div>
    </section>
  );
}

window.Contact = Contact;
```

---

## 12) `src/main.js`

```jsx
// main.js - assemble app and mount
const { useEffect } = React;

function App(){
  useEffect(()=>{
    // small reveal animation using motion.one if present
    try{ if(window.motion) window.motion.animate('section', { opacity:[0,1], translateY:[20,0] }, { delay:0.2, duration:0.7, easing:'ease-in-out', stagger:0.08 }); }catch(e){}
  },[]);

  return (
    <div>
      <Header />
      <main className="pt-20 bg-[#071021]">
        <Hero />
        <Services />
        <About />
        <Projects />
        <Team />
        <Contact />
        <footer className="py-8 border-t border-white/6 mt-12 text-center text-sm text-slate-400">© {new Date().getFullYear()} ODGLOBAL Limited — All rights reserved.</footer>
      </main>
    </div>
  );
}

ReactDOM.createRoot(document.getElementById('root')).render(<App />);
```

---

## Assets

* Add `assets/slide-1.jpg`, `slide-2.jpg`, `slide-3.jpg` (high-resolution oil & gas / industrial images). Use free images from Unsplash if needed, or your own corporate photos. Example Unsplash queries: `oil rig`, `industrial plant`, `engineering team`.
* Add `assets/logo.png` (your logo).

---

## How to deploy & push (quick commands)

From the root of your local project folder:

```bash
git add .
git commit -m "Enhanced site: separated components + parallax slideshow"
git push origin main
```

After pushing, GitHub Pages (if enabled for this repo) will update the site.

---

## Want me to:

* Produce a ZIP file of the above ready to drop into your repo? — say **"ZIP it"**
* Automatically create all files in the canvas for copy/paste? — say **"Create files in canvas"** (I already placed this doc in canvas)
* Replace Unsplash links with your corporate photos and partner logos? — attach images or say **"Use my images"**

---

That's everything — the files above will give you a clean, professional, fullscreen parallax slideshow hero (Option C) with separated React components and CDN-based buildless setup. Paste them into your repo, push, and your upgraded site will be live.
